terraform {
    backend s3 {}
    required_providers {
        aws = {
            source  = "hashicorp/aws"
            version = "~> 5.57.0"
        }
    }
}

# ネットワーク ##############################################################################################################
# VPC
resource aws_vpc vpc {
    cidr_block              = lookup(var.vpc_args, "cidr_block")
    instance_tenancy        = lookup(var.vpc_args, "instance_tenancy")
    enable_dns_support      = lookup(var.vpc_args, "enable_dns_support")
    enable_dns_hostnames    = lookup(var.vpc_args, "enable_dns_hostnames")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.vpc_args, "name_tag"))
    }
}
# サブネット
resource aws_subnet public_subnet_1a {
    vpc_id                  = aws_vpc.vpc.id
    cidr_block              = lookup(var.public_subnet_1a_args, "cidr_block")
    map_public_ip_on_launch = true
    availability_zone       = lookup(var.public_subnet_1a_args, "availability_zone")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.public_subnet_1a_args, "name_tag"))
    }
}
resource aws_subnet private_subnet_1a {
    vpc_id                  = aws_vpc.vpc.id
    cidr_block              = lookup(var.private_subnet_1a_args, "cidr_block")
    availability_zone       = lookup(var.private_subnet_1a_args, "availability_zone")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.private_subnet_1a_args, "name_tag"))
    }
}
resource aws_subnet public_subnet_1c {
    vpc_id                  = aws_vpc.vpc.id
    cidr_block              = lookup(var.public_subnet_1c_args, "cidr_block")
    map_public_ip_on_launch = true
    availability_zone       = lookup(var.public_subnet_1c_args, "availability_zone")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.public_subnet_1c_args, "name_tag"))
    }
}
resource aws_subnet private_subnet_1c {
    vpc_id                  = aws_vpc.vpc.id
    cidr_block              = lookup(var.private_subnet_1c_args, "cidr_block")
    availability_zone       = lookup(var.private_subnet_1c_args, "availability_zone")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.private_subnet_1c_args, "name_tag"))
    }
}

# インターネットゲートウェイ
resource aws_internet_gateway internet_gateway {
    vpc_id = aws_vpc.vpc.id
    tags = {
        Name = format("%s-%s", var.name_prefix, var.internet_gateway_name_tag)
    }
}
# ルートテーブル
resource aws_route_table route_table_to_internet {
    vpc_id = aws_vpc.vpc.id
    route {
        cidr_block = "0.0.0.0/0"
        gateway_id = aws_internet_gateway.internet_gateway.id
    }
    tags = {
        Name = format("%s-%s", var.name_prefix, var.route_table_to_internet_name_tag)
    }
}
# ルートテーブル 関連付け
resource aws_route_table_association route_association_public_subnet_1a {
    subnet_id       = aws_subnet.public_subnet_1a.id
    route_table_id  = aws_route_table.route_table_to_internet.id 
}
resource aws_route_table_association route_association_public_subnet_1c {
    subnet_id       = aws_subnet.public_subnet_1c.id
    route_table_id  = aws_route_table.route_table_to_internet.id 
}


# EFS ###################################################################################################################
resource "aws_efs_file_system" "efs" {
    performance_mode = lookup(var.efs_args, "performance_mode")
    throughput_mode  = lookup(var.efs_args, "throughput_mode")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.efs_args, "name_tag"))
    }
}
# マウントターゲット
resource "aws_efs_mount_target" "mount_target_1a" {
    file_system_id  = aws_efs_file_system.efs.id
    subnet_id       = aws_subnet.public_subnet_1a.id
    security_groups = [aws_security_group.security_group_for_efs.id]
}
resource "aws_efs_mount_target" "mount_target_1c" {
    file_system_id  = aws_efs_file_system.efs.id
    subnet_id       = aws_subnet.public_subnet_1c.id
    security_groups = [aws_security_group.security_group_for_efs.id]
}
# セキュリティグループ
resource aws_security_group security_group_for_efs {
    description = lookup(var.security_group_for_efs_args, "description")
    vpc_id      = aws_vpc.vpc.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_for_efs_args, "name_tag"))
    }
}
resource aws_vpc_security_group_ingress_rule ingress_rule_for_efs {
    from_port   = lookup(var.security_group_ingress_for_efs_args, "ingress_from_port")
    to_port     = lookup(var.security_group_ingress_for_efs_args, "ingress_to_port")
    ip_protocol = lookup(var.security_group_ingress_for_efs_args, "ingress_protocol")
    cidr_ipv4   = lookup(var.security_group_ingress_for_efs_args, "ingress_cidr_ipv4")
    referenced_security_group_id = aws_security_group.security_group_for_ec2.id
    security_group_id = aws_security_group.security_group_for_efs.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_ingress_for_efs_args, "name_tag"))
    }
}
resource aws_vpc_security_group_egress_rule egress_rule_for_efs {
    from_port   = lookup(var.security_group_egress_for_efs_args, "egress_from_port")
    to_port     = lookup(var.security_group_egress_for_efs_args, "egress_to_port")
    ip_protocol = lookup(var.security_group_egress_for_efs_args, "egress_protocol")
    cidr_ipv4   = lookup(var.security_group_egress_for_efs_args, "egress_cidr_ipv4")
    referenced_security_group_id = null
    security_group_id = aws_security_group.security_group_for_efs.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_egress_for_efs_args, "name_tag"))
    }
}

# EC2 ####################################################################################################################
# ec2 launch template
resource aws_launch_template ec2_launch_template {
    image_id                = lookup(var.ec2_launch_template_args, "ami_id")
    instance_type           = lookup(var.ec2_launch_template_args, "instance_type")
    user_data               = base64encode(templatefile("user_data_script.tpl", {
        efs_id = aws_efs_file_system.efs.id
        #efs_id = aws_efs_file_system.efs.dns_name
    }))
    block_device_mappings {
        device_name = lookup(var.ec2_launch_template_args, "block_device_name")
        ebs {
            volume_size             = lookup(var.ec2_launch_template_args, "ebs_volume_size")
            volume_type             = lookup(var.ec2_launch_template_args, "ebs_volume_type")
            delete_on_termination   = lookup(var.ec2_launch_template_args, "ebs_delete_on_termination")
        }
    } 
    tag_specifications {
        resource_type = "instance"
        tags = {
            Name = format(lookup(var.ec2_launch_template_args, "instance_name_tag"))
        }
    }
    iam_instance_profile {
        name = aws_iam_instance_profile.instance_profile.name
    }
    network_interfaces {
        associate_public_ip_address = true
        security_groups             = [aws_security_group.security_group_for_ec2.id]
    }
    tags = {
            Name = format(lookup(var.ec2_launch_template_args, "launch_template_name_tag"))
    }
    # efsのマウントをするため、マウントポイントが作成後EC2を作成する
    depends_on = [
        aws_efs_file_system.efs
    ]
}
# EC2用セキュリティグループ
resource aws_security_group security_group_for_ec2 {
    description = lookup(var.security_group_for_ec2_args, "description")
    vpc_id      = aws_vpc.vpc.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_for_ec2_args, "name_tag"))
    }
}
resource aws_vpc_security_group_ingress_rule ingress_rule_from_alb_for_ec2 {
    from_port   = lookup(var.security_group_ingress_from_alb_for_ec2_args, "ingress_from_port")
    to_port     = lookup(var.security_group_ingress_from_alb_for_ec2_args, "ingress_to_port")
    ip_protocol = lookup(var.security_group_ingress_from_alb_for_ec2_args, "ingress_protocol")
    cidr_ipv4   = lookup(var.security_group_ingress_from_alb_for_ec2_args, "ingress_cidr_ipv4")
    referenced_security_group_id = aws_security_group.security_group_for_alb.id
    security_group_id = aws_security_group.security_group_for_ec2.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_ingress_from_alb_for_ec2_args, "name_tag"))
    }
}
resource aws_vpc_security_group_ingress_rule ingress_rule_from_internet_for_ec2 {
    from_port   = lookup(var.security_group_ingress_from_internet_for_ec2_args, "ingress_from_port")
    to_port     = lookup(var.security_group_ingress_from_internet_for_ec2_args, "ingress_to_port")
    ip_protocol = lookup(var.security_group_ingress_from_internet_for_ec2_args, "ingress_protocol")
    cidr_ipv4   = lookup(var.security_group_ingress_from_internet_for_ec2_args, "ingress_cidr_ipv4")
    referenced_security_group_id = null
    security_group_id = aws_security_group.security_group_for_ec2.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_ingress_from_internet_for_ec2_args, "name_tag"))
    }
}
resource aws_vpc_security_group_egress_rule egress_rule_for_ec2 {
    from_port   = lookup(var.security_group_egress_for_ec2_args, "egress_from_port")
    to_port     = lookup(var.security_group_egress_for_ec2_args, "egress_to_port")
    ip_protocol = lookup(var.security_group_egress_for_ec2_args, "egress_protocol")
    cidr_ipv4   = lookup(var.security_group_egress_for_ec2_args, "egress_cidr_ipv4")
    referenced_security_group_id = null
    security_group_id = aws_security_group.security_group_for_ec2.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_egress_for_ec2_args, "name_tag"))
    }
}
# インスタンスプロファイル（インスタンスとIAM Roleの関連付け）
resource aws_iam_instance_profile instance_profile {
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.ec2_role_settings_args , "instance_profile_name_tag"))
    }
    role = aws_iam_role.instance_role.name
}
# IAM ロール
resource aws_iam_role instance_role {
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.ec2_role_settings_args , "iam_role_name_tag"))
    }
    assume_role_policy = data.aws_iam_policy_document.assume_role_policy.json
}
data aws_iam_policy_document assume_role_policy {
    statement {
        actions = ["sts:AssumeRole"]  
        principals {
            type        = "Service"
            identifiers = ["ec2.amazonaws.com"]
        }
    }
}
# SSM用ポリシー
resource aws_iam_role_policy_attachment attachment_ssm {
    role       = aws_iam_role.instance_role.name
    policy_arn = data.aws_iam_policy.ssm_policy.arn
}
data aws_iam_policy ssm_policy { 
    name = "AmazonSSMManagedInstanceCore"
}
# cloudwatchエージェント用ポリシー
resource aws_iam_role_policy_attachment attachment_cloudwatch_agent {
    role       = aws_iam_role.instance_role.name
    policy_arn = data.aws_iam_policy.cloudwatch_agent_policy.arn
}
data aws_iam_policy cloudwatch_agent_policy { 
    name = "CloudWatchAgentServerPolicy"
}

# ALB ####################################################################################
# alb
resource aws_lb alb {
    internal            = false
    load_balancer_type  = "application"
    security_groups     = [aws_security_group.security_group_for_alb.id]
    subnets             = [aws_subnet.public_subnet_1a.id, aws_subnet.private_subnet_1c.id]
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.alb_args, "name_tag"))
    }
}
# ターゲットグループ
resource aws_lb_target_group target_group {
    port        = 80
    protocol    = "HTTP"
    vpc_id      = aws_vpc.vpc.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.target_group_args, "name_tag"))
    }
    health_check {
        interval            = lookup(var.target_group_args, "health_check_interval")
        path                = lookup(var.target_group_args, "health_check_path")
        port                = "traffic-port"
        protocol            = "HTTP"
        timeout             = lookup(var.target_group_args, "health_check_timeout")
        healthy_threshold   = lookup(var.target_group_args, "healthy_threshold")
        unhealthy_threshold = lookup(var.target_group_args, "unhealthy_threshold")
    }
    stickiness {
        cookie_duration = lookup(var.target_group_args, "stickiness_cookie_duration")
        enabled         = lookup(var.target_group_args, "stickiness_enables")
        type            = lookup(var.target_group_args, "stickiness_type")
    }
}
# リスナー
resource aws_lb_listener listener {
    load_balancer_arn   = aws_lb.alb.arn
    port                = "80"
    protocol            = "HTTP"
    default_action {
        target_group_arn = aws_lb_target_group.target_group.arn
        type = "forward"
    }
    tags = {
        Name = format("%s-%s", var.name_prefix, var.listener_name_tag)
    }
}

# Auto Scaling Group
resource aws_autoscaling_group auto_scaling_group {
    name                        = format("%s-%s", var.name_prefix, lookup(var.auto_scaling_args, "name"))
    max_size                    = lookup(var.auto_scaling_args, "max_size")
    min_size                    = lookup(var.auto_scaling_args, "min_size")
    health_check_grace_period   = lookup(var.auto_scaling_args, "health_check_grace_period")
    health_check_type           = lookup(var.auto_scaling_args, "health_check_type")
    desired_capacity            = lookup(var.auto_scaling_args, "desired_capacity")
    vpc_zone_identifier         = [aws_subnet.public_subnet_1a.id, aws_subnet.public_subnet_1c.id]
    target_group_arns           = [aws_lb_target_group.target_group.arn]
    launch_template {
        id      = aws_launch_template.ec2_launch_template.id
        version = aws_launch_template.ec2_launch_template.latest_version
    }
}
# セキュリティグループ
resource aws_security_group security_group_for_alb {
    description = lookup(var.security_group_for_alb_args, "description")
    vpc_id      = aws_vpc.vpc.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_for_alb_args, "name_tag"))
    }
}
resource aws_vpc_security_group_ingress_rule ingress_rule_for_alb {
    from_port   = lookup(var.security_group_ingress_for_alb_args, "ingress_from_port")
    to_port     = lookup(var.security_group_ingress_for_alb_args, "ingress_to_port")
    ip_protocol = lookup(var.security_group_ingress_for_alb_args, "ingress_protocol")
    cidr_ipv4   = lookup(var.security_group_ingress_for_alb_args, "ingress_cidr_ipv4")
    referenced_security_group_id = null
    security_group_id = aws_security_group.security_group_for_alb.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_ingress_for_alb_args, "name_tag"))
    }
}
resource aws_vpc_security_group_egress_rule egress_rule_for_alb {
    from_port   = lookup(var.security_group_egress_for_alb_args, "egress_from_port")
    to_port     = lookup(var.security_group_egress_for_alb_args, "egress_to_port")
    ip_protocol = lookup(var.security_group_egress_for_alb_args, "egress_protocol")
    cidr_ipv4   = lookup(var.security_group_egress_for_alb_args, "egress_cidr_ipv4")
    referenced_security_group_id = null
    security_group_id = aws_security_group.security_group_for_alb.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_egress_for_alb_args, "name_tag"))
    }
}

# RDS ###########################################################################################################
# RDS DB インスタンス
resource aws_db_instance rds {
    allocated_storage       = lookup(var.rds_args, "allocated_storage")
    max_allocated_storage   = lookup(var.rds_args, "max_allocated_storage")
    storage_type            = lookup(var.rds_args, "storage_type")
    engine                  = lookup(var.rds_args, "engine")
    engine_version          = lookup(var.rds_args, "engine_version")
    instance_class          = lookup(var.rds_args, "instance_class")
    multi_az                = lookup(var.rds_args, "multi_az")
    skip_final_snapshot     = lookup(var.rds_args, "skip_final_snapshot")
    identifier              = lookup(var.rds_args, "identifier")
    db_name                 = lookup(var.rds_args, "db_name")
    username                = lookup(var.rds_args, "username")
    password                = lookup(var.rds_args, "password")
    vpc_security_group_ids  = [aws_security_group.security_group_for_rds.id]
    parameter_group_name    = aws_db_parameter_group.parameter_group.name
    option_group_name       = aws_db_option_group.option_group.name
    db_subnet_group_name    = aws_db_subnet_group.subnet_group.name
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.rds_args, "name_tag"))
    }
}
# パラメータグループ
resource aws_db_parameter_group parameter_group  {
    family = lookup(var.parameter_group_args, "family")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.parameter_group_args, "name_tag"))
    }
}

# オプショングループ
resource aws_db_option_group option_group {
    engine_name          = lookup(var.option_group_args, "engine_name")
    major_engine_version = lookup(var.option_group_args, "major_engine_version")
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.option_group_args, "name_tag"))
    }
}
# サブネットグループ
resource aws_db_subnet_group subnet_group {
    subnet_ids = [aws_subnet.private_subnet_1a.id, aws_subnet.private_subnet_1c.id]

    tags = {
        Name = format("%s-%s", var.name_prefix, var.subnet_group_name_tag)
    }
}
# セキュリティグループ
resource aws_security_group security_group_for_rds {
    description = lookup(var.security_group_for_rds_args, "description")
    vpc_id      = aws_vpc.vpc.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_for_rds_args, "name_tag"))
    }
}
resource aws_vpc_security_group_ingress_rule ingress_rule_for_rds {
    from_port   = lookup(var.security_group_ingress_for_rds_args, "ingress_from_port")
    to_port     = lookup(var.security_group_ingress_for_rds_args, "ingress_to_port")
    ip_protocol = lookup(var.security_group_ingress_for_rds_args, "ingress_protocol")
    cidr_ipv4   = lookup(var.security_group_ingress_for_rds_args, "ingress_cidr_ipv4")
    referenced_security_group_id = aws_security_group.security_group_for_ec2.id
    security_group_id = aws_security_group.security_group_for_rds.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_ingress_for_rds_args, "name_tag"))
    }
}
resource aws_vpc_security_group_egress_rule egress_rule_for_rds {
    from_port   = lookup(var.security_group_egress_for_rds_args, "egress_from_port")
    to_port     = lookup(var.security_group_egress_for_rds_args, "egress_to_port")
    ip_protocol = lookup(var.security_group_egress_for_rds_args, "egress_protocol")
    cidr_ipv4   = lookup(var.security_group_egress_for_rds_args, "egress_cidr_ipv4")
    referenced_security_group_id = null
    security_group_id = aws_security_group.security_group_for_rds.id
    tags = {
        Name = format("%s-%s", var.name_prefix, lookup(var.security_group_ingress_for_rds_args, "name_tag"))
    }
}