<?php
/**
 * Blog-Theme Template Functions Library
 * 
 * @package Cocoon Child
 * @subpackage Blog-Theme
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class BlogThemeTemplateFunctions {
    
    /**
     * Render post card
     * 
     * @param int $post_id Post ID
     * @param string $size Image size
     */
    public static function render_post_card($post_id = null, $size = 'thumbnail') {
        if (!$post_id) {
            $post_id = get_the_ID();
        }
        
        $post = get_post($post_id);
        if (!$post) {
            return;
        }
        ?>
        <div class="post-card">
            <div class="post-thumbnail">
                <?php if (has_post_thumbnail($post_id)) : ?>
                    <?php echo get_the_post_thumbnail($post_id, $size); ?>
                <?php else : ?>
                    <div class="no-thumbnail"></div>
                <?php endif; ?>
            </div>
            <div class="post-content">
                <h3><a href="<?php echo get_permalink($post_id); ?>"><?php echo get_the_title($post_id); ?></a></h3>
                <div class="post-meta">
                    <span class="post-date"><?php echo get_the_date('Y.m.d', $post_id); ?></span>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render section with posts
     * 
     * @param string $title Section title
     * @param WP_Query $query Posts query
     * @param bool $show_more Show more button
     * @param string $more_url More button URL
     */
    public static function render_posts_section($title, $query, $show_more = true, $more_url = '') {
        ?>
        <div class="section">
            <div class="section-header">
                <h2><?php echo esc_html($title); ?></h2>
            </div>
            <div class="posts-slider">
                <?php
                if ($query->have_posts()) :
                    while ($query->have_posts()) : $query->the_post();
                        self::render_post_card();
                    endwhile;
                    wp_reset_postdata();
                else :
                    echo '<p class="no-posts">投稿が見つかりませんでした。</p>';
                endif;
                ?>
            </div>
            <?php if ($show_more && !empty($more_url)) : ?>
            <div class="section-footer">
                <a href="<?php echo esc_url($more_url); ?>" class="more-btn">もっと見る</a>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render breadcrumb
     */
    public static function render_breadcrumb() {
        if (is_front_page()) {
            return;
        }
        
        echo '<nav class="breadcrumb">';
        echo '<a href="' . home_url() . '">ホーム</a>';
        
        if (is_category()) {
            $category = get_queried_object();
            echo ' > <span>' . $category->name . '</span>';
        } elseif (is_single()) {
            $categories = get_the_category();
            if (!empty($categories)) {
                $category = $categories[0];
                echo ' > <a href="' . get_category_link($category->term_id) . '">' . $category->name . '</a>';
            }
            echo ' > <span>' . get_the_title() . '</span>';
        } elseif (is_page()) {
            echo ' > <span>' . get_the_title() . '</span>';
        }
        
        echo '</nav>';
    }
    
    /**
     * Render pagination
     * 
     * @param WP_Query $query Query object
     */
    public static function render_pagination($query = null) {
        if (!$query) {
            global $wp_query;
            $query = $wp_query;
        }
        
        $total_pages = $query->max_num_pages;
        
        if ($total_pages <= 1) {
            return;
        }
        
        $current_page = max(1, get_query_var('paged'));
        
        echo '<nav class="pagination">';
        
        // Previous page link
        if ($current_page > 1) {
            echo '<a href="' . get_pagenum_link($current_page - 1) . '" class="prev">前へ</a>';
        }
        
        // Page numbers
        for ($i = 1; $i <= $total_pages; $i++) {
            if ($i == $current_page) {
                echo '<span class="current">' . $i . '</span>';
            } else {
                echo '<a href="' . get_pagenum_link($i) . '">' . $i . '</a>';
            }
        }
        
        // Next page link
        if ($current_page < $total_pages) {
            echo '<a href="' . get_pagenum_link($current_page + 1) . '" class="next">次へ</a>';
        }
        
        echo '</nav>';
    }
    
    /**
     * Render social share buttons
     * 
     * @param int $post_id Post ID
     */
    public static function render_social_share($post_id = null) {
        if (!$post_id) {
            $post_id = get_the_ID();
        }
        
        $url = get_permalink($post_id);
        $title = get_the_title($post_id);
        
        ?>
        <div class="social-share">
            <h4>この記事をシェア</h4>
            <div class="share-buttons">
                <a href="<?php echo BlogThemeUtilities::get_social_share_url('twitter', $url, $title); ?>" 
                   target="_blank" class="share-twitter">Twitter</a>
                <a href="<?php echo BlogThemeUtilities::get_social_share_url('facebook', $url, $title); ?>" 
                   target="_blank" class="share-facebook">Facebook</a>
                <a href="<?php echo BlogThemeUtilities::get_social_share_url('line', $url, $title); ?>" 
                   target="_blank" class="share-line">LINE</a>
            </div>
        </div>
        <?php
    }
}