# CloudFormation パラメータドキュメント

このドキュメントは、Apex Domain Redirect システムで使用されるすべての CloudFormation パラメータに関する詳細情報を提供します。

## 概要

CloudFormation テンプレートは、デプロイ設定を制御する6つのパラメータを受け付けます。3つのパラメータは必須（ApexDomain、MigrationTargetFQDN、CertificateArn）で、3つはデフォルト値を持つオプションパラメータです。

## パラメータリファレンス

### ApexDomain

**説明**: 移行先にリダイレクトされる Apex ドメイン名。

**型**: String

**必須**: はい

**デフォルト**: なし

**目的**: このパラメータは、ユーザーがアクセスするソースドメインを指定します。Lambda 関数は、この値をリダイレクトページに元のドメイン名として表示するために使用します。

**値の例**:

- `example.com`
- `mycompany.co.jp`
- `mydomain.org`

**検証**:

- 有効なドメイン名形式である必要があります
- プロトコル（http:// または https://）を含めないでください
- サブドメインを含めないでください（Apex ドメインのみを使用）
- 末尾のスラッシュを含めないでください

**テンプレート内での使用**:

- Lambda 関数に環境変数として渡されます
- 識別用のリソースタグに使用されます
- HTML リダイレクトページに表示されます

**注意事項**:

- これは外部 DNS プロバイダーで設定するドメインです
- このドメインを Elastic IP に向ける A レコードを作成します

---

### MigrationTargetFQDN

**説明**: ユーザーがリダイレクトされる移行先ウェブサイトの完全修飾ドメイン名（FQDN）。

**型**: String

**必須**: はい

**デフォルト**: なし

**目的**: このパラメータは、リダイレクトの宛先 URL を指定します。Lambda 関数は、10秒後にユーザーをこの FQDN に自動的にリダイレクトする HTML ページを生成します。

**値の例**:

- `www.example.com`
- `d1234567890abc.cloudfront.net`（CloudFront ディストリビューション）
- `newsite.example.com`
- `example.s3-website-ap-northeast-1.amazonaws.com`（S3 静的ウェブサイト）

**検証**:

- 有効な FQDN 形式である必要があります
- プロトコル（http:// または https://）を含めないでください
- パスやクエリパラメータを含めないでください
- 末尾のスラッシュを含めないでください

**テンプレート内での使用**:

- Lambda 関数に環境変数として渡されます
- HTML 内のリダイレクト URL を構築するために使用されます（https:// が自動的に付加されます）
- HTML リダイレクトページに表示されます

**注意事項**:

- Lambda 関数は、リダイレクト URL でこの FQDN の前に `https://` を付加します
- ターゲットウェブサイトが HTTPS をサポートしていることを確認してください
- このリダイレクトをデプロイする前に、ターゲットウェブサイトがアクセス可能で動作していることを確認してください

---

### CertificateArn

**説明**: Application Load Balancer での HTTPS/SSL 終端のための、ap-northeast-1 リージョンの ACM（AWS Certificate Manager）証明書の ARN（Amazon Resource Name）。

**型**: String

**必須**: はい

**デフォルト**: なし

**目的**: このパラメータは、Application Load Balancer が SSL/TLS 終端に使用する ACM 証明書を指定します。証明書により HTTPS 接続が可能になり、クライアントとリダイレクトサービス間の安全な通信が保証されます。

**値の例**:

- `arn:aws:acm:ap-northeast-1:123456789012:certificate/12345678-1234-1234-1234-123456789012`

**検証**:

- 有効な ACM 証明書 ARN である必要があります
- ap-northeast-1 リージョン（ALB と同じ）である必要があります
- 証明書のステータスが「発行済み」である必要があります
- 証明書がリダイレクトされる Apex ドメインをカバーしている必要があります
- 形式: `arn:aws:acm:ap-northeast-1:[0-9]{12}:certificate/[a-f0-9-]+`

**テンプレート内での使用**:

- ALB HTTPS リスナーによって SSL/TLS 終端のために参照されます
- HTTPS（ポート443）接続を有効にします
- 証明書が正しいリージョンにあることを検証します

**注意事項**:

- ACM 証明書はリージョン固有のリソースです
- 証明書は Application Load Balancer と同じリージョン（ap-northeast-1）にある必要があります
- 他のリージョン（例: CloudFront 用の us-east-1）で作成された証明書は使用できません
- 新しい証明書をリクエストするか、既存の証明書をインポートできます
- 証明書発行には DNS 検証が推奨されます
- 証明書は Apex ドメイン（例: example.com）をカバーしている必要があります

**証明書の準備**:

新しい証明書をリクエスト:
```bash
aws acm request-certificate \
  --domain-name example.com \
  --validation-method DNS \
  --region ap-northeast-1
```

既存の証明書を一覧表示:
```bash
aws acm list-certificates --region ap-northeast-1 --output table
```

証明書のステータスを確認:
```bash
aws acm describe-certificate \
  --certificate-arn <YOUR_CERTIFICATE_ARN> \
  --region ap-northeast-1 \
  --query 'Certificate.{Domain:DomainName,Status:Status}' \
  --output table
```

---

### ResourcePrefix

**説明**: 識別と整理のために、すべてのリソース名の前に付加されるプレフィックス文字列。

**型**: String

**必須**: いいえ

**デフォルト**: `kodama-apexdomain`

**目的**: このパラメータは、特に複数のスタックをデプロイする場合や、異なるプロジェクト間でリソースを管理する場合に、AWS アカウント内のリソースを整理および識別するのに役立ちます。

**値の例**:

- `mycompany-redirect`
- `prod-apex-redirect`
- `staging-domain-redirect`
- `team-migration`

**検証**:

- 英数字とハイフンのみを含む必要があります
- ハイフンで始まったり終わったりしてはいけません
- 1〜32文字である必要があります
- AWS リソース命名規則に従う必要があります

**テンプレート内での使用**:

- リソース名の前に付加されます（例: `{prefix}-nlb`、`{prefix}-alb`）
- すべてのリソースにタグとして追加されます
- CloudFormation 出力で使用されます

**影響を受けるリソース**:

- VPC: `{prefix}-vpc`
- サブネット: `{prefix}-public-subnet-1`、`{prefix}-private-subnet-1` など
- インターネットゲートウェイ: `{prefix}-igw`
- ルートテーブル: `{prefix}-public-rt`
- セキュリティグループ: `{prefix}-alb-sg`
- Network Load Balancer: `{prefix}-nlb`
- Application Load Balancer: `{prefix}-alb`
- Lambda 関数: `{prefix}-redirect-function`
- IAM ロール: `{prefix}-lambda-role`
- WAF WebACL: `{prefix}-waf-webacl`

**注意事項**:

- 目的と環境を識別するのに役立つ意味のあるプレフィックスを選択してください
- 関連リソース間で一貫したプレフィックスを使用してください
- プレフィックスに環境（prod、staging）またはチーム名を含めることを検討してください

---

### CostTag

**説明**: 請求とコスト追跡の目的で使用されるコスト配分タグ値。

**型**: String

**必須**: いいえ

**デフォルト**: `j.kodama`

**目的**: このパラメータは、AWS Cost Explorer と請求レポートでのコスト追跡と配分を可能にします。これらのリソースに関連するコストに責任を持つチーム、プロジェクト、または個人を識別するのに役立ちます。

**値の例**:

- `engineering-team`
- `project-migration`
- `john.doe`
- `cost-center-1234`
- `department-it`

**検証**:

- コスト追跡のための意味のある識別子である必要があります
- 英数字、スペース、一般的な句読点を含めることができます
- 組織のコスト配分タグ標準に従う必要があります

**テンプレート内での使用**:

- すべてのタグ付け可能なリソースに `Cost` タグとして適用されます
- AWS Cost Explorer でコストのフィルタリングとグループ化に使用されます
- コスト配分のための請求レポートに表示されます

**タグ付けされるリソース**:

- VPC
- サブネット
- インターネットゲートウェイ
- Elastic IP
- Network Load Balancer
- Application Load Balancer
- セキュリティグループ
- Lambda 関数
- IAM ロール
- WAF WebACL

**注意事項**:

- 正しい値については、財務または運用チームと調整してください
- すべての AWS リソースで一貫したコストタグを使用してください
- レポートで使用するには、AWS Billing コンソールでコスト配分タグを有効にしてください
- このタグは「これらのコストに誰が責任を持つか？」という質問に答えるのに役立ちます

---

### CreateDateTag

**説明**: リソースが作成された日時を示すタグ値。リソースライフサイクル管理と監査に使用されます。

**型**: String

**必須**: いいえ

**デフォルト**: `2025/12/26`

**目的**: このパラメータは、リソースがいつ作成されたかを追跡するのに役立ちます。これは、監査、コンプライアンス、ライフサイクル管理、および古いまたは未使用のリソースの識別に役立ちます。

**値の例**:

- `2025/01/21`
- `2025-01-21`
- `January 21, 2025`
- `Q1-2025`
- `2025/01/21-migration-phase1`

**検証**:

- 組織にとって意味のある任意の文字列形式を使用できます
- 推奨形式: 一貫性とソート可能性のために `YYYY/MM/DD`
- 必要に応じて追加のコンテキストを含めることができます

**テンプレート内での使用**:

- すべてのタグ付け可能なリソースに `CreateDate` タグとして適用されます
- リソースライフサイクル管理に使用されます
- AWS Resource Groups でリソースの経過時間を識別するのに役立ちます

**タグ付けされるリソース**:

- VPC
- サブネット
- インターネットゲートウェイ
- Elastic IP
- Network Load Balancer
- Application Load Balancer
- セキュリティグループ
- Lambda 関数
- IAM ロール
- WAF WebACL

**注意事項**:

- 国際的な一貫性のために ISO 8601 形式（YYYY-MM-DD）の使用を検討してください
- スタックを再デプロイまたは更新する際は、この値を更新してください
- レビューまたはクリーンアップが必要なリソースを識別するのに役立ちます
- コンプライアンスチェックのために AWS Config ルールで使用できます

---

## パラメータ使用例

### 例1: 本番環境デプロイ

```bash
aws cloudformation create-stack \
  --stack-name apex-redirect-prod \
  --template-body file://apex-domain-redirect-main.yaml \
  --parameters \
    ParameterKey=ApexDomain,ParameterValue=example.com \
    ParameterKey=MigrationTargetFQDN,ParameterValue=www.example.com \
    ParameterKey=CertificateArn,ParameterValue=arn:aws:acm:ap-northeast-1:123456789012:certificate/12345678-1234-1234-1234-123456789012 \
    ParameterKey=ResourcePrefix,ParameterValue=prod-redirect \
    ParameterKey=CostTag,ParameterValue=engineering-team \
    ParameterKey=CreateDateTag,ParameterValue=2025/01/21 \
  --capabilities CAPABILITY_NAMED_IAM \
  --region ap-northeast-1
```

### 例2: ステージング環境デプロイ

```bash
aws cloudformation create-stack \
  --stack-name apex-redirect-staging \
  --template-body file://apex-domain-redirect-main.yaml \
  --parameters \
    ParameterKey=ApexDomain,ParameterValue=staging.example.com \
    ParameterKey=MigrationTargetFQDN,ParameterValue=d1234567890abc.cloudfront.net \
    ParameterKey=CertificateArn,ParameterValue=arn:aws:acm:ap-northeast-1:123456789012:certificate/87654321-4321-4321-4321-210987654321 \
    ParameterKey=ResourcePrefix,ParameterValue=staging-redirect \
    ParameterKey=CostTag,ParameterValue=qa-team \
    ParameterKey=CreateDateTag,ParameterValue=2025/01/21 \
  --capabilities CAPABILITY_NAMED_IAM \
  --region ap-northeast-1
```

### 例3: パラメータファイルの使用

パラメータファイル `parameters.json` を作成:

```json
[
  {
    "ParameterKey": "ApexDomain",
    "ParameterValue": "example.com"
  },
  {
    "ParameterKey": "MigrationTargetFQDN",
    "ParameterValue": "www.example.com"
  },
  {
    "ParameterKey": "CertificateArn",
    "ParameterValue": "arn:aws:acm:ap-northeast-1:123456789012:certificate/12345678-1234-1234-1234-123456789012"
  },
  {
    "ParameterKey": "ResourcePrefix",
    "ParameterValue": "mycompany-redirect"
  },
  {
    "ParameterKey": "CostTag",
    "ParameterValue": "migration-project"
  },
  {
    "ParameterKey": "CreateDateTag",
    "ParameterValue": "2025/01/21"
  }
]
```

パラメータファイルを使用してデプロイ:

```bash
aws cloudformation create-stack \
  --stack-name apex-domain-redirect \
  --template-body file://apex-domain-redirect-main.yaml \
  --parameters file://parameters.json \
  --capabilities CAPABILITY_IAM \
  --region ap-northeast-1
```

## パラメータのベストプラクティス

### 1. ApexDomain

- 所有および管理している実際の Apex ドメインを使用してください
- デプロイ前にドメインの所有権を確認してください
- DNS プロバイダーが A レコードをサポートしていることを確認してください
- 設定後に DNS 解決をテストしてください

### 2. MigrationTargetFQDN

- デプロイ前にターゲットウェブサイトが動作していることを確認してください
- ターゲットが HTTPS をサポートしていることを確認してください
- リダイレクトで使用する前に、ターゲット URL を手動でテストしてください
- パフォーマンスと HTTPS サポートの向上のために CloudFront の使用を検討してください

### 3. CertificateArn

- 証明書が ap-northeast-1 リージョン（ALB と同じ）にあることを確認してください
- デプロイ前に証明書のステータスが「発行済み」であることを確認してください
- 証明書が Apex ドメインをカバーしていることを確認してください
- より迅速な証明書発行のために DNS 検証を使用してください
- テストには、提供されているテスト証明書 ARN を使用してください

### 4. ResourcePrefix

- 環境と目的を示す説明的なプレフィックスを使用してください
- プレフィックスは短く、しかし意味のあるものにしてください（例: `prod-redirect`、`p-r` ではなく）
- 組織全体で一貫した命名規則を使用してください
- ハイフン以外の特殊文字は避けてください

### 5. CostTag

- 承認されたコストセンターコードについては、財務チームと調整してください
- すべての AWS リソースで一貫したコストタグを使用してください
- AWS Billing コンソールでコスト配分タグを有効にしてください
- 適切な配分を確保するために、コストレポートを定期的に確認してください

### 6. CreateDateTag

- 一貫性のために ISO 8601 日付形式（YYYY-MM-DD）を使用してください
- スタックに重要な変更を加える際は、日付を更新してください
- 役立つ場合は追加のコンテキストを含めてください（例: `2025/01/21-v2`）
- コンプライアンスのために AWS Config ルールでこのタグを使用してください

## パラメータ検証

CloudFormation テンプレートには基本的なパラメータ検証が含まれていますが、デプロイ前にパラメータを検証する必要があります:

### デプロイ前チェックリスト

- [ ] ApexDomain が有効なドメイン名形式である
- [ ] ApexDomain にプロトコルやサブドメインが含まれていない
- [ ] MigrationTargetFQDN がアクセス可能で動作している
- [ ] MigrationTargetFQDN が HTTPS をサポートしている
- [ ] CertificateArn が ap-northeast-1 リージョンにある
- [ ] CertificateArn の証明書のステータスが「発行済み」である
- [ ] 証明書が Apex ドメインをカバーしている
- [ ] ResourcePrefix が命名規則に従っている
- [ ] CostTag が組織の標準に一致している
- [ ] CreateDateTag が正しい形式である

### 検証スクリプトの例

```bash
#!/bin/bash

# ApexDomain 形式の検証
if [[ ! "$APEX_DOMAIN" =~ ^[a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?(\.[a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?)*$ ]]; then
  echo "エラー: 無効な ApexDomain 形式"
  exit 1
fi

# MigrationTargetFQDN がアクセス可能かどうかの検証
if ! curl -s -o /dev/null -w "%{http_code}" "https://$MIGRATION_TARGET_FQDN" | grep -q "^[23]"; then
  echo "警告: MigrationTargetFQDN にアクセスできない可能性があります"
fi

# ResourcePrefix 形式の検証
if [[ ! "$RESOURCE_PREFIX" =~ ^[a-z0-9]([a-z0-9-]{0,30}[a-z0-9])?$ ]]; then
  echo "エラー: 無効な ResourcePrefix 形式"
  exit 1
fi

echo "パラメータ検証に合格しました"
```

## パラメータの更新

デプロイ後にパラメータを更新するには:

```bash
aws cloudformation update-stack \
  --stack-name apex-domain-redirect \
  --use-previous-template \
  --parameters \
    ParameterKey=ApexDomain,UsePreviousValue=true \
    ParameterKey=MigrationTargetFQDN,ParameterValue=new-target.example.com \
    ParameterKey=CertificateArn,UsePreviousValue=true \
    ParameterKey=ResourcePrefix,UsePreviousValue=true \
    ParameterKey=CostTag,UsePreviousValue=true \
    ParameterKey=CreateDateTag,ParameterValue=2025/01/22 \
  --capabilities CAPABILITY_NAMED_IAM \
  --region ap-northeast-1
```

**注意**: 一部のパラメータ（ResourcePrefix など）を変更すると、リソースの置き換えが発生し、ダウンタイムが発生する可能性があります。更新を適用する前に、変更セットを確認してください。

## トラブルシューティング

### 問題: "Parameter validation failed"

**原因**: パラメータ値が CloudFormation の制約を満たしていない

**解決策**:

- パラメータの形式と制約を確認してください
- 末尾のスペースや特殊文字がないことを確認してください
- パラメータ値が期待されるパターンと一致することを確認してください

### 問題: "Invalid domain name"

**原因**: ApexDomain または MigrationTargetFQDN に無効な文字が含まれている

**解決策**:

- プロトコル（http:// または https://）を削除してください
- パスやクエリパラメータを削除してください
- 末尾のスラッシュを削除してください
- 有効なドメイン文字（a-z、0-9、ハイフン、ドット）のみを使用してください

### 問題: "Resource names too long"

**原因**: ResourcePrefix が長すぎて、リソース名が AWS の制限を超えている

**解決策**:

- より短い ResourcePrefix を使用してください（推奨: 10〜20文字）
- プレフィックス内の冗長な単語を避けてください

## 関連ドキュメント

- [README.md](README.md) - メインプロジェクトドキュメントとデプロイガイド
- [apex-domain-redirect-main.yaml](apex-domain-redirect-main.yaml) - CloudFormation テンプレート
- [.kiro/specs/apex-domain-redirect/requirements.md](.kiro/specs/apex-domain-redirect/requirements.md) - 要件仕様
- [.kiro/specs/apex-domain-redirect/design.md](.kiro/specs/apex-domain-redirect/design.md) - 設計ドキュメント
