# Apex Domain Redirect

AWS Lambda、Application Load Balancer、Network Load Balancer、AWS WAF を使用したサーバーレス Apex ドメインリダイレクトシステム。このソリューションは、DNS プロバイダーが A レコードのみをサポートし（ALIAS 機能なし）、Apex ドメイン用の高可用性マルチ AZ リダイレクトサービスを提供します。

## 目次

- [概要](#概要)
- [アーキテクチャ](#アーキテクチャ)
- [前提条件](#前提条件)
- [プロジェクト構造](#プロジェクト構造)
- [CloudFormation パラメータ](#cloudformation-パラメータ)
- [デプロイガイド](#デプロイガイド)
- [DNS 設定](#dns-設定)
- [テスト手順](#テスト手順)
- [監視とトラブルシューティング](#監視とトラブルシューティング)
- [コスト見積もり](#コスト見積もり)
- [クリーンアップ](#クリーンアップ)

## 概要

このシステムは、サーバーレスアーキテクチャを使用して、Apex ドメイン（例: example.com）からターゲットウェブサイト（例: www.example.com または CloudFront ディストリビューション）へのトラフィックをリダイレクトします。Lambda 関数は、10秒のカウントダウンと自動リダイレクトを含む HTML ページを生成します。

主な機能:

- マルチ AZ 高可用性（99.99% SLA 目標）
- DNS A レコード用の固定 IP アドレス（Elastic IP）
- 一般的な Web 攻撃に対する AWS WAF 保護
- コスト効率のためのサーバーレスアーキテクチャ
- CloudFormation を使用した Infrastructure as Code

## アーキテクチャ

システムは多層アプローチを使用します:

```
外部 DNS (A レコード) → Elastic IP (2 AZ) → NLB → ALB → Lambda 関数
                                                      ↓
                                                   AWS WAF
```

コンポーネント:

- **Elastic IP**: DNS A レコード用の2つの静的 IP アドレス（AZ ごとに1つ）
- **Network Load Balancer (NLB)**: Elastic IP サポート付きレイヤー4ロードバランシング
- **Application Load Balancer (ALB)**: Lambda へのレイヤー7 HTTP ルーティング
- **AWS WAF**: SQL インジェクション、XSS、レート制限に対する保護
- **Lambda 関数**: カウントダウン付き HTML リダイレクトレスポンスを生成

## 前提条件

このソリューションをデプロイする前に、以下を確認してください:

### AWS 要件

1. **AWS アカウント** と適切な権限:
   - CloudFormation: フルアクセス
   - EC2: VPC、サブネット、セキュリティグループ、Elastic IP の作成
   - Elastic Load Balancing: ALB と NLB の作成と管理
   - Lambda: 関数の作成と管理
   - IAM: ロールとポリシーの作成
   - WAF: WebACL の作成と管理
   - CloudWatch: ロググループとメトリクスの作成
   - ACM: 証明書へのアクセス

2. **ACM 証明書** (ap-northeast-1 リージョン):
   - 証明書のステータスが「発行済み」である必要があります
   - 証明書が Apex ドメインをカバーしている必要があります
   - 証明書 ARN 形式: `arn:aws:acm:ap-northeast-1:ACCOUNT_ID:certificate/CERTIFICATE_ID`

3. **AWS CLI** のインストールと設定:

   ```bash
   aws --version
   # AWS CLI バージョン 2.x 以上が表示されるはずです
   ```

3. **AWS プロファイル** の設定（MFA を使用する場合）:

   ```bash
   export AWS_PROFILE=your-profile-name
   aws sts get-caller-identity
   ```

4. **ACM 証明書の確認**（必要に応じて）:

   ```bash
   # ap-northeast-1 の証明書一覧を表示
   aws acm list-certificates --region ap-northeast-1 --output table
   
   # 特定の証明書の詳細を表示
   aws acm describe-certificate \
     --certificate-arn <YOUR_CERTIFICATE_ARN> \
     --region ap-northeast-1
   ```

### DNS 要件

1. 外部 DNS プロバイダーの管理コンソールへのアクセス
2. Apex ドメインの A レコードを作成する機能
3. DNS TTL 設定機能（推奨: 300秒）

### ローカルツール

1. **cfn-lint**（オプション、テンプレート検証用）:

   ```bash
   pip install cfn-lint
   ```

2. **jq**（オプション、JSON 出力の解析用）:

   ```bash
   # macOS
   brew install jq
   
   # Linux
   sudo apt-get install jq
   ```

## プロジェクト構造

```
.
├── apex-domain-redirect-main.yaml        # メイン CloudFormation テンプレート
├── test-environment.yaml                 # オプションのテスト環境テンプレート
├── src/
│   └── lambda/
│       └── index.py                       # Lambda 関数コード（開発/テスト用）
│                                          # 注: 実際のデプロイは CloudFormation テンプレート内のインラインコードを使用
├── scripts/
│   ├── deploy.sh                          # デプロイスクリプト
│   ├── integration-test.sh                # 統合テストスクリプト
│   └── validate-template.sh               # テンプレート検証スクリプト
├── tests/                                 # テストファイル
│   ├── test_lambda_function.py            # Lambda ユニットテスト
│   ├── test_lambda_property_based.py      # Lambda プロパティベーステスト
│   ├── test_cloudformation_template_property_based.py  # テンプレート PBT
│   ├── test_resource_tagging_property_based.py         # タグ付け PBT
│   ├── test_waf_property_based.py         # WAF PBT
│   ├── test_environment_unit_tests.py     # テスト環境ユニットテスト
│   ├── integration-test-results.log       # 統合テストログ
│   └── ...                                # その他のテストファイル
├── deploy_test_environment.sh             # テスト環境デプロイスクリプト
├── PARAMETERS.md                          # パラメータドキュメント
├── PARAMETERS-jp.md                       # パラメータドキュメント（日本語）
├── INTEGRATION_TEST_PLAN.md               # 統合テスト計画
├── INTEGRATION_TEST_RESULTS.md            # 統合テスト結果
└── README.md                              # このファイル（英語版）
```

**Lambda 関数コードに関する注意:**

- Lambda 関数コードは CloudFormation テンプレート（`apex-domain-redirect-main.yaml`）内に**インライン**でデプロイされます
- `src/lambda/index.py` は**開発とテストの目的のみ**で提供されています
- これにより、ローカルテストとコードレビューが容易になりますが、デプロイ時に直接使用されることはありません
- 実際にデプロイされるコードは、CloudFormation テンプレートの `Code.ZipFile` セクションに埋め込まれています

## CloudFormation パラメータ

詳細なパラメータドキュメントについては、[PARAMETERS-jp.md](PARAMETERS-jp.md) を参照してください。

クイックリファレンス:

| パラメータ | 必須 | デフォルト | 説明 |
|-----------|------|-----------|------|
| ApexDomain | はい | - | Apex ドメイン名（例: example.com） |
| MigrationTargetFQDN | はい | - | ターゲットウェブサイト FQDN（例: www.example.com） |
| CertificateArn | はい | - | ap-northeast-1 リージョンの ACM 証明書 ARN |
| ResourcePrefix | いいえ | kodama-apexdomain | リソース名のプレフィックス |
| CostTag | いいえ | j.kodama | コスト配分タグ値 |
| CreateDateTag | いいえ | 2025/12/26 | リソース作成日タグ |

## デプロイガイド

### ステップ1: CloudFormation テンプレートの検証

デプロイ前に、テンプレートの構文を検証します:

```bash
# AWS CLI を使用
aws cloudformation validate-template \
  --template-body file://apex-domain-redirect-main.yaml

# cfn-lint を使用（インストールされている場合）
cfn-lint apex-domain-redirect-main.yaml
```

### ステップ2: CloudFormation スタックのデプロイ

#### オプション A: AWS CLI を使用

```bash
aws cloudformation create-stack \
  --stack-name apex-domain-redirect \
  --template-body file://apex-domain-redirect-main.yaml \
  --parameters \
    ParameterKey=ApexDomain,ParameterValue=example.com \
    ParameterKey=MigrationTargetFQDN,ParameterValue=www.example.com \
    ParameterKey=CertificateArn,ParameterValue=arn:aws:acm:ap-northeast-1:ACCOUNT_ID:certificate/CERTIFICATE_ID \
    ParameterKey=ResourcePrefix,ParameterValue=mycompany-redirect \
    ParameterKey=CostTag,ParameterValue=myteam \
    ParameterKey=CreateDateTag,ParameterValue=2025/01/21 \
  --capabilities CAPABILITY_NAMED_IAM \
  --region ap-northeast-1
```

#### オプション B: デプロイスクリプトを使用

```bash
./deploy.sh \
  --stack-name apex-domain-redirect \
  --apex-domain example.com \
  --target-fqdn www.example.com \
  --certificate-arn arn:aws:acm:ap-northeast-1:ACCOUNT_ID:certificate/CERTIFICATE_ID \
  --prefix mycompany-redirect \
  --cost-tag myteam \
  --region ap-northeast-1
```

### ステップ3: スタック作成の監視

スタック作成の進行状況を監視します:

```bash
aws cloudformation describe-stack-events \
  --stack-name apex-domain-redirect \
  --region ap-northeast-1 \
  --max-items 20
```

または AWS コンソールを使用:

1. AWS コンソールで CloudFormation に移動
2. スタックを選択
3. 「イベント」タブをクリックして進行状況を監視

スタックの作成には通常5〜10分かかります。

### ステップ4: Elastic IP アドレスの取得

スタックが作成されたら、Elastic IP アドレスを取得します:

```bash
aws cloudformation describe-stacks \
  --stack-name apex-domain-redirect \
  --region ap-northeast-1 \
  --query 'Stacks[0].Outputs' \
  --output table
```

`ElasticIP1` と `ElasticIP2` の値をメモしてください - DNS 設定に必要です。

## DNS 設定

デプロイ後、外部 DNS プロバイダーを設定します:

### ステップ1: A レコードの作成

Apex ドメインに2つの A レコードを作成します:

```
タイプ: A
名前: @ (または Apex ドメイン)
値: <CloudFormation 出力の ElasticIP1>
TTL: 300

タイプ: A
名前: @ (または Apex ドメイン)
値: <CloudFormation 出力の ElasticIP2>
TTL: 300
```

### ステップ2: DNS 伝播の確認

DNS 伝播を待ち（通常5〜15分）、確認します:

```bash
# DNS 解決を確認
dig example.com +short

# 両方の Elastic IP アドレスが返されるはずです
# 203.0.113.1
# 203.0.113.2

# 複数の場所から確認
nslookup example.com 8.8.8.8
```

### DNS プロバイダーの例

#### Route 53（AWS DNS を使用する場合）

```bash
aws route53 change-resource-record-sets \
  --hosted-zone-id Z1234567890ABC \
  --change-batch file://dns-changes.json
```

#### その他の DNS プロバイダー

- **Cloudflare**: DNS → レコードを追加 → タイプ: A、名前: @、コンテンツ: <IP>
- **GoDaddy**: DNS 管理 → 追加 → タイプ: A、ホスト: @、ポイント先: <IP>
- **Namecheap**: 高度な DNS → 新しいレコードを追加 → タイプ: A レコード、ホスト: @、値: <IP>

## テスト手順

### テスト1: NLB への直接アクセス

NLB に直接テストします（DNS 設定前）:

```bash
# CloudFormation 出力から NLB DNS 名を取得
NLB_DNS=$(aws cloudformation describe-stacks \
  --stack-name apex-domain-redirect \
  --region ap-northeast-1 \
  --query 'Stacks[0].Outputs[?OutputKey==`NLBDNSName`].OutputValue' \
  --output text)

# HTTP リクエストをテスト
curl -v http://$NLB_DNS

# リダイレクトページを含む HTML が返されるはずです
```

### テスト2: HTML レスポンスの確認

HTML に正しいリダイレクト情報が含まれていることを確認します:

```bash
curl -s http://$NLB_DNS | grep -E "(meta.*refresh|window.location.href|サイト移転)"
```

期待される出力には以下が含まれるはずです:

- 10秒遅延のメタリフレッシュタグ
- JavaScript リダイレクトコード
- 移転通知テキスト

### テスト3: Apex ドメイン経由のテスト

DNS 設定後、Apex ドメイン経由でテストします:

```bash
curl -v http://example.com

# またはブラウザを使用して完全なリダイレクト体験を確認
```

### テスト4: マルチ AZ 機能の確認

両方の Elastic IP が応答していることをテストします:

```bash
# Elastic IP 1 をテスト
curl -v http://<ElasticIP1>

# Elastic IP 2 をテスト
curl -v http://<ElasticIP2>

# 両方とも同じリダイレクトページを返すはずです
```

### テスト5: ブラウザテスト

1. ブラウザを開き、Apex ドメイン（例: http://example.com）に移動
2. リダイレクトページが以下を表示することを確認:
   - 移転通知メッセージ
   - ターゲット URL の表示
   - カウントダウンタイマー（10秒）
3. 自動リダイレクトを待つか、「今すぐ移動する」ボタンをクリック
4. ターゲット URL にリダイレクトされることを確認

### テスト6: WAF 保護

WAF が悪意のあるリクエストをブロックしていることをテストします:

```bash
# SQL インジェクションをテスト（ブロックされるはずです）
curl -v "http://example.com/?id=1' OR '1'='1"

# XSS をテスト（ブロックされるはずです）
curl -v "http://example.com/?search=<script>alert('xss')</script>"

# CloudWatch で WAF メトリクスを確認
aws cloudwatch get-metric-statistics \
  --namespace AWS/WAFV2 \
  --metric-name BlockedRequests \
  --dimensions Name=Rule,Value=ALL \
  --start-time $(date -u -d '1 hour ago' +%Y-%m-%dT%H:%M:%S) \
  --end-time $(date -u +%Y-%m-%dT%H:%M:%S) \
  --period 3600 \
  --statistics Sum \
  --region ap-northeast-1
```

## 監視とトラブルシューティング

### CloudWatch ログ

Lambda 関数のログを表示:

```bash
# ロググループ名を取得
LOG_GROUP="/aws/lambda/kodama-apexdomain-redirect-function"

# 最近のログを表示
aws logs tail $LOG_GROUP --follow --region ap-northeast-1
```

### CloudWatch メトリクス

主要なメトリクスを監視:

```bash
# Lambda 呼び出し
aws cloudwatch get-metric-statistics \
  --namespace AWS/Lambda \
  --metric-name Invocations \
  --dimensions Name=FunctionName,Value=kodama-apexdomain-redirect-function \
  --start-time $(date -u -d '1 hour ago' +%Y-%m-%dT%H:%M:%S) \
  --end-time $(date -u +%Y-%m-%dT%H:%M:%S) \
  --period 300 \
  --statistics Sum \
  --region ap-northeast-1

# ALB ターゲットレスポンスタイム
aws cloudwatch get-metric-statistics \
  --namespace AWS/ApplicationELB \
  --metric-name TargetResponseTime \
  --dimensions Name=LoadBalancer,Value=<ALB-ARN-suffix> \
  --start-time $(date -u -d '1 hour ago' +%Y-%m-%dT%H:%M:%S) \
  --end-time $(date -u +%Y-%m-%dT%H:%M:%S) \
  --period 300 \
  --statistics Average \
  --region ap-northeast-1
```

### よくある問題

#### 問題: "Elastic IP limit exceeded" でスタック作成が失敗する

**解決策**: AWS Service Quotas コンソールで Elastic IP の制限引き上げをリクエストしてください。

#### 問題: DNS が Elastic IP に解決されない

**解決策**:

1. DNS プロバイダーで A レコードが正しく設定されていることを確認
2. DNS 伝播を確認: `dig example.com +short`
3. TTL の有効期限を待つ（通常5〜15分）

#### 問題: ALB から 502 Bad Gateway

**解決策**:

1. Lambda 関数のログでエラーを確認
2. Lambda 関数に正しい環境変数があることを確認
3. ALB ターゲットグループのヘルスステータスを確認

#### 問題: WAF が正当なトラフィックをブロックしている

**解決策**:

1. CloudWatch で WAF ログを確認
2. 誤検知ルールを特定
3. 例外を追加するか、ルールの感度を調整

## コスト見積もり

東京リージョン（ap-northeast-1）での月間推定コスト（月間100万リクエスト）:

| サービス | コスト |
|---------|------|
| Elastic IP (2個) | $7.20 |
| Network Load Balancer | $22.00 + データ処理 |
| Application Load Balancer | $22.00 + LCU 料金 |
| Lambda (100万リクエスト、128MB、100ms) | $0.20 |
| AWS WAF | $6.60 |
| CloudWatch ログとメトリクス | $5.00 |
| データ転送 | 変動 |
| **合計** | **約 $65-75/月** |

コストは以下によって変動する可能性があります:

- リクエスト量
- レスポンスサイズ
- データ転送アウト
- CloudWatch ログ保持期間

## クリーンアップ

すべてのリソースを削除して課金を停止するには:

### ステップ1: CloudFormation スタックの削除

```bash
aws cloudformation delete-stack \
  --stack-name apex-domain-redirect \
  --region ap-northeast-1
```

### ステップ2: 削除の監視

```bash
aws cloudformation describe-stack-events \
  --stack-name apex-domain-redirect \
  --region ap-northeast-1 \
  --max-items 20
```

### ステップ3: DNS レコードの削除

外部 DNS プロバイダーから A レコードを削除します。

### ステップ4: クリーンアップの確認

すべてのリソースが削除されたことを確認します:

```bash
aws cloudformation describe-stacks \
  --stack-name apex-domain-redirect \
  --region ap-northeast-1

# "Stack with id apex-domain-redirect does not exist" が返されるはずです
```

### Elastic IP に関する注意

Elastic IP は、スタックが削除されると自動的に解放されます。削除が失敗した場合は、手動で解放してください:

```bash
aws ec2 describe-addresses --region ap-northeast-1
aws ec2 release-address --allocation-id <allocation-id> --region ap-northeast-1
```

## サポートと貢献

問題、質問、または貢献については、`.kiro/specs/apex-domain-redirect/` のプロジェクトドキュメントを参照してください。

## ライセンス

このプロジェクトは、内部使用のために現状のまま提供されています。
