#!/bin/bash

# Apex Domain Redirect - Deployment Script
# This script deploys the CloudFormation stack with parameter validation

set -e  # Exit on error

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Default values
STACK_NAME=""
APEX_DOMAIN=""
TARGET_FQDN=""
CERTIFICATE_ARN=""
RESOURCE_PREFIX="kodama-apexdomain"
COST_TAG="j.kodama"
CREATE_DATE_TAG=$(date +%Y/%m/%d)
REGION="ap-northeast-1"
TEMPLATE_FILE="apex-domain-redirect-main.yaml"
VALIDATE_ONLY=false
UPDATE_STACK=false

# Function to print colored output
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Function to display usage
usage() {
    cat << EOF
Usage: $0 [OPTIONS]

Deploy the Apex Domain Redirect CloudFormation stack.

Required Options:
    --stack-name NAME           CloudFormation stack name
    --apex-domain DOMAIN        Apex domain name (e.g., example.com)
    --target-fqdn FQDN          Migration target FQDN (e.g., www.example.com)
    --certificate-arn ARN       ACM Certificate ARN in ap-northeast-1 region

Optional Options:
    --prefix PREFIX             Resource prefix (default: kodama-apexdomain)
    --cost-tag TAG              Cost allocation tag (default: j.kodama)
    --create-date DATE          Creation date tag (default: current date)
    --region REGION             AWS region (default: ap-northeast-1)
    --template FILE             Template file path (default: apex-domain-redirect-main.yaml)
    --validate-only             Only validate template, don't deploy
    --update                    Update existing stack instead of creating new one
    -h, --help                  Display this help message

Examples:
    # Create new stack
    $0 --stack-name my-redirect \\
       --apex-domain example.com \\
       --target-fqdn www.example.com \\
       --certificate-arn arn:aws:acm:ap-northeast-1:123456789012:certificate/12345678-1234-1234-1234-123456789012

    # Create stack with custom prefix and tags
    $0 --stack-name prod-redirect \\
       --apex-domain example.com \\
       --target-fqdn www.example.com \\
       --certificate-arn arn:aws:acm:ap-northeast-1:123456789012:certificate/12345678-1234-1234-1234-123456789012 \\
       --prefix prod-redirect \\
       --cost-tag engineering-team

    # Update existing stack
    $0 --stack-name my-redirect \\
       --apex-domain example.com \\
       --target-fqdn new-target.example.com \\
       --certificate-arn arn:aws:acm:ap-northeast-1:123456789012:certificate/12345678-1234-1234-1234-123456789012 \\
       --update

    # Validate template only
    $0 --validate-only

EOF
    exit 1
}

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --stack-name)
            STACK_NAME="$2"
            shift 2
            ;;
        --apex-domain)
            APEX_DOMAIN="$2"
            shift 2
            ;;
        --target-fqdn)
            TARGET_FQDN="$2"
            shift 2
            ;;
        --certificate-arn)
            CERTIFICATE_ARN="$2"
            shift 2
            ;;
        --prefix)
            RESOURCE_PREFIX="$2"
            shift 2
            ;;
        --cost-tag)
            COST_TAG="$2"
            shift 2
            ;;
        --create-date)
            CREATE_DATE_TAG="$2"
            shift 2
            ;;
        --region)
            REGION="$2"
            shift 2
            ;;
        --template)
            TEMPLATE_FILE="$2"
            shift 2
            ;;
        --validate-only)
            VALIDATE_ONLY=true
            shift
            ;;
        --update)
            UPDATE_STACK=true
            shift
            ;;
        -h|--help)
            usage
            ;;
        *)
            print_error "Unknown option: $1"
            usage
            ;;
    esac
done

# Function to validate domain format
validate_domain() {
    local domain=$1
    local domain_regex='^[a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?(\.[a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?)*$'
    
    if [[ ! "$domain" =~ $domain_regex ]]; then
        return 1
    fi
    
    # Check for invalid patterns
    if [[ "$domain" =~ ^https?:// ]] || [[ "$domain" =~ / ]]; then
        return 1
    fi
    
    return 0
}

# Function to validate resource prefix
validate_prefix() {
    local prefix=$1
    local prefix_regex='^[a-z0-9]([a-z0-9-]{0,30}[a-z0-9])?$'
    
    if [[ ! "$prefix" =~ $prefix_regex ]]; then
        return 1
    fi
    
    return 0
}

# Function to check if template file exists
check_template_file() {
    if [[ ! -f "$TEMPLATE_FILE" ]]; then
        print_error "Template file not found: $TEMPLATE_FILE"
        exit 1
    fi
    print_success "Template file found: $TEMPLATE_FILE"
}

# Function to validate CloudFormation template
validate_template() {
    print_info "Validating CloudFormation template..."
    
    if aws cloudformation validate-template \
        --template-body file://"$TEMPLATE_FILE" \
        --region "$REGION" > /dev/null 2>&1; then
        print_success "Template validation passed"
        return 0
    else
        print_error "Template validation failed"
        aws cloudformation validate-template \
            --template-body file://"$TEMPLATE_FILE" \
            --region "$REGION"
        return 1
    fi
}

# Function to check if stack exists
stack_exists() {
    aws cloudformation describe-stacks \
        --stack-name "$STACK_NAME" \
        --region "$REGION" > /dev/null 2>&1
    return $?
}

# Function to create CloudFormation stack
create_stack() {
    print_info "Creating CloudFormation stack: $STACK_NAME"
    
    aws cloudformation create-stack \
        --stack-name "$STACK_NAME" \
        --template-body file://"$TEMPLATE_FILE" \
        --parameters \
            ParameterKey=ApexDomain,ParameterValue="$APEX_DOMAIN" \
            ParameterKey=MigrationTargetFQDN,ParameterValue="$TARGET_FQDN" \
            ParameterKey=CertificateArn,ParameterValue="$CERTIFICATE_ARN" \
            ParameterKey=ResourcePrefix,ParameterValue="$RESOURCE_PREFIX" \
            ParameterKey=CostTag,ParameterValue="$COST_TAG" \
            ParameterKey=CreateDateTag,ParameterValue="$CREATE_DATE_TAG" \
        --capabilities CAPABILITY_NAMED_IAM \
        --region "$REGION" \
        --tags \
            Key=Name,Value="$STACK_NAME" \
            Key=ManagedBy,Value=CloudFormation \
            Key=Purpose,Value=ApexDomainRedirect
    
    if [[ $? -eq 0 ]]; then
        print_success "Stack creation initiated"
        return 0
    else
        print_error "Stack creation failed"
        return 1
    fi
}

# Function to update CloudFormation stack
update_stack() {
    print_info "Updating CloudFormation stack: $STACK_NAME"
    
    aws cloudformation update-stack \
        --stack-name "$STACK_NAME" \
        --template-body file://"$TEMPLATE_FILE" \
        --parameters \
            ParameterKey=ApexDomain,ParameterValue="$APEX_DOMAIN" \
            ParameterKey=MigrationTargetFQDN,ParameterValue="$TARGET_FQDN" \
            ParameterKey=CertificateArn,ParameterValue="$CERTIFICATE_ARN" \
            ParameterKey=ResourcePrefix,ParameterValue="$RESOURCE_PREFIX" \
            ParameterKey=CostTag,ParameterValue="$COST_TAG" \
            ParameterKey=CreateDateTag,ParameterValue="$CREATE_DATE_TAG" \
        --capabilities CAPABILITY_NAMED_IAM \
        --region "$REGION"
    
    if [[ $? -eq 0 ]]; then
        print_success "Stack update initiated"
        return 0
    else
        # Check if no updates are needed
        if aws cloudformation describe-stacks --stack-name "$STACK_NAME" --region "$REGION" > /dev/null 2>&1; then
            print_warning "No updates to be performed (stack may already be up to date)"
            return 0
        else
            print_error "Stack update failed"
            return 1
        fi
    fi
}

# Function to wait for stack operation to complete
wait_for_stack() {
    local operation=$1
    print_info "Waiting for stack $operation to complete..."
    print_info "This may take 5-10 minutes. You can monitor progress in the AWS Console."
    
    if [[ "$operation" == "create" ]]; then
        aws cloudformation wait stack-create-complete \
            --stack-name "$STACK_NAME" \
            --region "$REGION"
    else
        aws cloudformation wait stack-update-complete \
            --stack-name "$STACK_NAME" \
            --region "$REGION"
    fi
    
    if [[ $? -eq 0 ]]; then
        print_success "Stack $operation completed successfully"
        return 0
    else
        print_error "Stack $operation failed or timed out"
        print_info "Check stack events for details:"
        print_info "aws cloudformation describe-stack-events --stack-name $STACK_NAME --region $REGION"
        return 1
    fi
}

# Function to display stack outputs
display_outputs() {
    print_info "Retrieving stack outputs..."
    
    local outputs=$(aws cloudformation describe-stacks \
        --stack-name "$STACK_NAME" \
        --region "$REGION" \
        --query 'Stacks[0].Outputs' \
        --output table)
    
    if [[ $? -eq 0 ]]; then
        echo ""
        print_success "Stack Outputs:"
        echo "$outputs"
        echo ""
        
        # Extract Elastic IPs
        local eip1=$(aws cloudformation describe-stacks \
            --stack-name "$STACK_NAME" \
            --region "$REGION" \
            --query 'Stacks[0].Outputs[?OutputKey==`ElasticIP1`].OutputValue' \
            --output text)
        
        local eip2=$(aws cloudformation describe-stacks \
            --stack-name "$STACK_NAME" \
            --region "$REGION" \
            --query 'Stacks[0].Outputs[?OutputKey==`ElasticIP2`].OutputValue' \
            --output text)
        
        if [[ -n "$eip1" ]] && [[ -n "$eip2" ]]; then
            print_info "Next Steps:"
            echo ""
            echo "1. Configure DNS A records for your apex domain ($APEX_DOMAIN):"
            echo "   - A record: $APEX_DOMAIN → $eip1"
            echo "   - A record: $APEX_DOMAIN → $eip2"
            echo "   - TTL: 300 seconds"
            echo ""
            echo "2. Wait for DNS propagation (5-15 minutes)"
            echo ""
            echo "3. Test the redirect:"
            echo "   curl -v http://$APEX_DOMAIN"
            echo ""
        fi
    else
        print_warning "Could not retrieve stack outputs"
    fi
}

# Main execution
main() {
    echo ""
    print_info "Apex Domain Redirect - Deployment Script"
    echo ""
    
    # Check if validate-only mode
    if [[ "$VALIDATE_ONLY" == true ]]; then
        check_template_file
        validate_template
        exit $?
    fi
    
    # Validate required parameters
    if [[ -z "$STACK_NAME" ]] || [[ -z "$APEX_DOMAIN" ]] || [[ -z "$TARGET_FQDN" ]] || [[ -z "$CERTIFICATE_ARN" ]]; then
        print_error "Missing required parameters"
        usage
    fi
    
    # Validate domain formats
    print_info "Validating parameters..."
    
    if ! validate_domain "$APEX_DOMAIN"; then
        print_error "Invalid apex domain format: $APEX_DOMAIN"
        print_info "Domain should not include protocol (http://) or paths"
        exit 1
    fi
    print_success "Apex domain format valid: $APEX_DOMAIN"
    
    if ! validate_domain "$TARGET_FQDN"; then
        print_error "Invalid target FQDN format: $TARGET_FQDN"
        print_info "FQDN should not include protocol (http://) or paths"
        exit 1
    fi
    print_success "Target FQDN format valid: $TARGET_FQDN"
    
    if ! validate_prefix "$RESOURCE_PREFIX"; then
        print_error "Invalid resource prefix format: $RESOURCE_PREFIX"
        print_info "Prefix should contain only lowercase letters, numbers, and hyphens"
        exit 1
    fi
    print_success "Resource prefix format valid: $RESOURCE_PREFIX"
    
    # Validate certificate ARN format
    if [[ ! "$CERTIFICATE_ARN" =~ ^arn:aws:acm:ap-northeast-1:[0-9]{12}:certificate/[a-f0-9-]+$ ]]; then
        print_error "Invalid certificate ARN format: $CERTIFICATE_ARN"
        print_info "Certificate ARN must be in ap-northeast-1 region"
        print_info "Format: arn:aws:acm:ap-northeast-1:123456789012:certificate/12345678-1234-1234-1234-123456789012"
        exit 1
    fi
    print_success "Certificate ARN format valid"
    
    # Check template file
    check_template_file
    
    # Validate template
    if ! validate_template; then
        exit 1
    fi
    
    # Display deployment summary
    echo ""
    print_info "Deployment Summary:"
    echo "  Stack Name:       $STACK_NAME"
    echo "  Apex Domain:      $APEX_DOMAIN"
    echo "  Target FQDN:      $TARGET_FQDN"
    echo "  Certificate ARN:  $CERTIFICATE_ARN"
    echo "  Resource Prefix:  $RESOURCE_PREFIX"
    echo "  Cost Tag:         $COST_TAG"
    echo "  Create Date:      $CREATE_DATE_TAG"
    echo "  Region:           $REGION"
    echo "  Template:         $TEMPLATE_FILE"
    echo ""
    
    # Check if stack exists
    if stack_exists; then
        if [[ "$UPDATE_STACK" == true ]]; then
            print_warning "Stack already exists. Updating..."
            if update_stack; then
                wait_for_stack "update"
                display_outputs
            else
                exit 1
            fi
        else
            print_error "Stack already exists: $STACK_NAME"
            print_info "Use --update flag to update the existing stack"
            exit 1
        fi
    else
        if [[ "$UPDATE_STACK" == true ]]; then
            print_error "Stack does not exist: $STACK_NAME"
            print_info "Remove --update flag to create a new stack"
            exit 1
        else
            if create_stack; then
                wait_for_stack "create"
                display_outputs
            else
                exit 1
            fi
        fi
    fi
    
    echo ""
    print_success "Deployment completed successfully!"
    echo ""
}

# Run main function
main
