#!/bin/bash

# Deploy Test Environment for Apex Domain Redirect System
# This script deploys the CloudFront + S3 test environment
# Requirements: Optional 1.1, 1.2, 1.3, 1.4, 1.5

set -e

# ============================================================
# DEPLOYMENT CONFIGURATION - Edit these values as needed
# ============================================================

# AWS Configuration
REGION="ap-northeast-1"

# AWS Profile for cli. Set "" for CloudShell.
PROFILE="your-aws-profile-name"

# Stack Configuration
STACK_NAME="kodama-apxdmntst-env"

# Resource Configuration
RESOURCE_PREFIX="kodama-apxdmn-tst"
COST_TAG="j.kodama"

# S3 Bucket Configuration
# Note: Bucket suffix will be auto-generated (8 random characters)
# Final bucket name will be: ${RESOURCE_PREFIX}-website-${BUCKET_SUFFIX}
# Example: kodama-apexdomain-test-website-a1b2c3d4

# ============================================================
# DO NOT EDIT BELOW THIS LINE (unless you know what you're doing)
# ============================================================

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
TEMPLATE_FILE="$SCRIPT_DIR/test-environment.yaml"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Function to check if AWS CLI is configured
check_aws_cli() {
    if ! command -v aws &> /dev/null; then
        print_error "AWS CLI is not installed. Please install AWS CLI first."
        exit 1
    fi
    
    if ! aws sts get-caller-identity --profile "$PROFILE" &> /dev/null; then
        print_error "AWS CLI profile '$PROFILE' is not configured or not accessible."
        print_error "Please configure your AWS credentials first."
        exit 1
    fi
    
    print_success "AWS CLI is properly configured"
}

# Function to validate CloudFormation template
validate_template() {
    print_status "Validating CloudFormation template..."
    
    if aws cloudformation validate-template \
        --template-body "file://$TEMPLATE_FILE" \
        --profile "$PROFILE" \
        --region "$REGION" > /dev/null; then
        print_success "CloudFormation template is valid"
    else
        print_error "CloudFormation template validation failed"
        exit 1
    fi
}

# Function to validate and show template (debug mode)
validate_and_show_template() {
    print_status "Validating CloudFormation template..."
    
    # Validate template
    if aws cloudformation validate-template \
        --template-body "file://$TEMPLATE_FILE" \
        --profile "$PROFILE" \
        --region "$REGION" > /dev/null; then
        print_success "CloudFormation template is valid"
    else
        print_error "CloudFormation template validation failed"
        exit 1
    fi
    
    # Show what parameters would be used
    local bucket_suffix
    bucket_suffix=$(generate_bucket_suffix)
    
    echo ""
    print_status "Parameters that would be used for deployment:"
    echo "  ResourcePrefix: $RESOURCE_PREFIX"
    echo "  CostTag: $COST_TAG"
    echo "  CreateDate: $(date +%Y/%m/%d)"
    echo "  BucketSuffix: $bucket_suffix"
    echo ""
    
    # Show expected resource names
    print_status "Expected resource names:"
    echo "  S3 Bucket: ${RESOURCE_PREFIX}-website-$bucket_suffix"
    echo "  IAM Role: ${RESOURCE_PREFIX}-s3-objects-role"
    echo "  CloudFront Distribution: (auto-generated)"
    echo ""
    
    # Show critical IAM policy resources that will be created
    print_status "Critical IAM Policy Resources (Debug Info):"
    echo "  S3ObjectsFunctionRole Policy Resources:"
    echo "    - arn:aws:s3:::${RESOURCE_PREFIX}-website-$bucket_suffix/*"
    echo "    - arn:aws:s3:::${RESOURCE_PREFIX}-website-$bucket_suffix"
    echo ""
    echo "  BucketPolicy Resource:"
    echo "    - arn:aws:s3:::${RESOURCE_PREFIX}-website-$bucket_suffix/*"
    echo ""
    
    # Show template validation details
    print_status "Template validation details:"
    aws cloudformation validate-template \
        --template-body "file://$TEMPLATE_FILE" \
        --profile "$PROFILE" \
        --region "$REGION" \
        --output table
    
    echo ""
    print_status "Template content preview :"
    echo "=================================================="
    cat "$TEMPLATE_FILE"
    echo "=================================================="
    echo ""
    print_warning "Use 'cat $TEMPLATE_FILE' to see the full template content"
}

# Function to check if stack exists
stack_exists() {
    aws cloudformation describe-stacks \
        --stack-name "$STACK_NAME" \
        --profile "$PROFILE" \
        --region "$REGION" \
        --query 'Stacks[0].StackStatus' \
        --output text 2>/dev/null || echo "DOES_NOT_EXIST"
}

# Function to generate bucket suffix
generate_bucket_suffix() {
    # Generate a random alphanumeric suffix (8 characters, lowercase only)
    local suffix
    suffix=$(openssl rand -hex 4)
    echo "$suffix"
}

# Function to deploy stack
deploy_stack() {
    local stack_status
    stack_status=$(stack_exists)
    
    # Generate bucket suffix
    local bucket_suffix
    bucket_suffix=$(generate_bucket_suffix)
    
    if [ "$stack_status" = "DOES_NOT_EXIST" ]; then
        print_status "Creating new CloudFormation stack: $STACK_NAME"
        print_status "Using bucket suffix: $bucket_suffix"
        
        aws cloudformation create-stack \
            --stack-name "$STACK_NAME" \
            --template-body "file://$TEMPLATE_FILE" \
            --capabilities CAPABILITY_NAMED_IAM \
            --parameters \
                ParameterKey=ResourcePrefix,ParameterValue=$RESOURCE_PREFIX \
                ParameterKey=CostTag,ParameterValue=$COST_TAG \
                ParameterKey=CreateDate,ParameterValue=$(date +%Y/%m/%d) \
                ParameterKey=BucketSuffix,ParameterValue=$bucket_suffix \
            --tags \
                Key=Purpose,Value=TestEnvironment \
                Key=ManagedBy,Value=CloudFormation \
                Key=Project,Value=ApexDomainRedirect \
            --profile "$PROFILE" \
            --region "$REGION"
        
        print_status "Waiting for stack creation to complete..."
        aws cloudformation wait stack-create-complete \
            --stack-name "$STACK_NAME" \
            --profile "$PROFILE" \
            --region "$REGION"
        
        print_success "Stack created successfully"
    else
        print_status "Updating existing CloudFormation stack: $STACK_NAME"
        print_status "Using bucket suffix: $bucket_suffix"
        
        aws cloudformation update-stack \
            --stack-name "$STACK_NAME" \
            --template-body "file://$TEMPLATE_FILE" \
            --capabilities CAPABILITY_NAMED_IAM \
            --parameters \
                ParameterKey=ResourcePrefix,ParameterValue=$RESOURCE_PREFIX \
                ParameterKey=CostTag,ParameterValue=$COST_TAG \
                ParameterKey=CreateDate,ParameterValue=$(date +%Y/%m/%d) \
                ParameterKey=BucketSuffix,ParameterValue=$bucket_suffix \
            --profile "$PROFILE" \
            --region "$REGION" 2>/dev/null || {
            print_warning "No updates to be performed on the stack"
            return 0
        }
        
        print_status "Waiting for stack update to complete..."
        aws cloudformation wait stack-update-complete \
            --stack-name "$STACK_NAME" \
            --profile "$PROFILE" \
            --region "$REGION"
        
        print_success "Stack updated successfully"
    fi
}

# Function to get stack outputs
get_stack_outputs() {
    print_status "Retrieving stack outputs..."
    
    local outputs
    outputs=$(aws cloudformation describe-stacks \
        --stack-name "$STACK_NAME" \
        --profile "$PROFILE" \
        --region "$REGION" \
        --query 'Stacks[0].Outputs' \
        --output table)
    
    echo "$outputs"
    
    # Get specific important outputs
    local cloudfront_domain
    cloudfront_domain=$(aws cloudformation describe-stacks \
        --stack-name "$STACK_NAME" \
        --profile "$PROFILE" \
        --region "$REGION" \
        --query 'Stacks[0].Outputs[?OutputKey==`CloudFrontDomainName`].OutputValue' \
        --output text)
    
    local test_url
    test_url=$(aws cloudformation describe-stacks \
        --stack-name "$STACK_NAME" \
        --profile "$PROFILE" \
        --region "$REGION" \
        --query 'Stacks[0].Outputs[?OutputKey==`TestWebsiteURL`].OutputValue' \
        --output text)
    
    echo ""
    print_success "Test environment deployed successfully!"
    echo ""
    echo "📋 Important Information:"
    echo "  CloudFront Domain: $cloudfront_domain"
    echo "  Test Website URL:  $test_url"
    echo ""
    echo "🔗 Usage Instructions:"
    echo "  1. Use '$cloudfront_domain' as the RedirectTargetFQDN parameter"
    echo "     in the main apex domain redirect CloudFormation template"
    echo "  2. Test the website by visiting: $test_url"
    echo "  3. Test redirect functionality by deploying the main template"
    echo ""
    echo "🧪 Test URLs:"
    echo "  Main page:    $test_url"
    echo "  Sample page:  $test_url/sample.html"
    echo "  404 page:     $test_url/404.html"
}

# Function to clean up (delete stack)
cleanup() {
    print_warning "This will delete the test environment stack: $STACK_NAME"
    read -p "Are you sure you want to continue? (y/N): " -n 1 -r
    echo
    
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        print_status "Deleting CloudFormation stack: $STACK_NAME"
        
        aws cloudformation delete-stack \
            --stack-name "$STACK_NAME" \
            --profile "$PROFILE" \
            --region "$REGION"
        
        print_status "Waiting for stack deletion to complete..."
        aws cloudformation wait stack-delete-complete \
            --stack-name "$STACK_NAME" \
            --profile "$PROFILE" \
            --region "$REGION"
        
        print_success "Stack deleted successfully"
    else
        print_status "Cleanup cancelled"
    fi
}

# Main function
main() {
    echo "=================================================="
    echo "  Apex Domain Redirect - Test Environment Deploy"
    echo "=================================================="
    echo ""
    
    case "${1:-deploy}" in
        "deploy")
            check_aws_cli
            validate_template
            deploy_stack
            get_stack_outputs
            ;;
        "cleanup"|"delete")
            check_aws_cli
            cleanup
            ;;
        "status"|"info")
            check_aws_cli
            get_stack_outputs
            ;;
        "validate"|"debug")
            check_aws_cli
            validate_and_show_template
            ;;
        "help"|"-h"|"--help")
            echo "Usage: $0 [command]"
            echo ""
            echo "Commands:"
            echo "  deploy    Deploy or update the test environment (default)"
            echo "  validate  Validate template and show content (debug mode)"
            echo "  cleanup   Delete the test environment stack"
            echo "  status    Show current stack status and outputs"
            echo "  help      Show this help message"
            echo ""
            echo "Configuration:"
            echo "  Stack Name: $STACK_NAME"
            echo "  Region:     $REGION"
            echo "  Profile:    $PROFILE"
            ;;
        *)
            print_error "Unknown command: $1"
            echo "Use '$0 help' for usage information"
            exit 1
            ;;
    esac
}

# Run main function with all arguments
main "$@"